# Webniam Payment PHP SDK

This SDK provides a lightweight, fully validated wrapper around the Webniam Payment API that powers `index.php` in this project. It mirrors all supported actions, applies the same validation rules used by the production gateway, and performs POST requests with JSON payloads.

## Installation

Include the SDK folder in your project and register the autoloader:

```php
require __DIR__ . '/sdk/autoload.php';
```

The SDK requires PHP 8.1+ with the cURL extension enabled.

## Getting Started

```php
use WebniamPayment\Client;

$client = new Client(
    'https://payments.example.com/index.php', // API endpoint
    'BUSINESS-ID',                            // business_id
    'YOUR-API-KEY'                            // api_key
);
```

`business_id` and `api_key` are stored on the client and injected into every request automatically. Each method returns an `ApiResponse` object with helpers for transport status and decoded body access.

```php
$response = $client->getBusinessInfo();

if ($response->isApiSuccess()) {
    $business = $response->getBody()['business'];
}
```

## Action Coverage

All actions exposed by `index.php` have first-class wrappers. The SDK performs local validation before issuing the HTTP request, mirroring the checks in the production endpoint.

### Business
- `validateBusiness(?string $environment = null)`
- `getBusinessInfo(?string $environment = null)`
- `getAllUsers()`
- `registerBusiness(array $newBusiness)`
- `modifyBusiness(array $updates)`
- `changeApiKey(?string $targetBusinessId = null)`
- `addStripeCredentials(array $credentials)`
- `modifyStripeCredentials(array $credentials)`
- `addAuthorizeCredentials(array $credentials)`
- `modifyAuthorizeCredentials(array $credentials)`
- `getStripePublicKey(string $environment = 'TEST')`

### Users & Cards
- `validateUser(?string $userId = null, ?string $email = null)`
- `getUserInfo(?string $userId = null, ?string $email = null)`
- `getAllUsers()`
- `getUserCards(?string $userId = null, ?string $email = null, string $source = 'both')`
- `getUserPrimaryCard(?string $userId = null, ?string $email = null, string $source = 'both')`
- `createUser(array $newUser)`
- `updateUser(string $userId, array $updates)`

### Payments
- `chargeCustomer(array $options)`
- `fundCustomer(array $options)`
- `createStripeCustomer(array $customerData)`
- `createAuthorizeCustomer(array $customerData)`
- `getPaymentRecords(?string $userId, ?string $email, array $filters = [])`

### Wallets
- `getUserWallet(?string $userId = null, ?string $email = null)`
- `getWalletHistory(?string $userId = null, ?string $email = null, int $limit = 50)`
- `chargeWallet(?string $userId, ?string $email, $amount, ?string $reason = null, ?string $description = null)`
- `addWalletFunds(?string $userId, ?string $email, $amount, ?string $reason = null, ?string $description = null)`

### Recurring Billing
- `createRecurring(?string $userId, ?string $email, $amount, array $options = [])`
- `listRecurring(?string $userId, ?string $email, int $page = 1, int $limit = 25)`
- `updateRecurring(string $recurringId, array $updates)`
- `pauseRecurring(string $recurringId)`
- `resumeRecurring(string $recurringId)`
- `cancelRecurring(string $recurringId)`
- `getRecurringLogs(string $recurringId, int $limit = 50)`

## Handling Responses

Each call returns `WebniamPayment\Response\ApiResponse`:

```php
$response = $client->chargeCustomer([
    'user_id' => 'USR-123',
    'amount' => 12.75,
    'currency' => 'USD',
    'description' => 'Espresso bar tab', // automatically base64-encoded
]);

if ($response->isApiSuccess()) {
    // Access decoded JSON body
    $data = $response->getBody();
} else {
    // Inspect status or raw JSON
    $status = $response->getStatusCode();
    $raw = $response->getRawBody();
}
```

## Custom HTTP Client

The SDK uses a cURL-based client by default. You can provide your own implementation of `HttpClientInterface` to integrate with frameworks or logging systems:

```php
$client->setHttpClient(new YourCustomHttpClient());
```

## Validation Errors

When a payload does not meet the API requirements, a `ValidationException` is thrown before the HTTP request is dispatched. This mirrors the server-side validation in `index.php` and helps catch mistakes early.

## Error Handling

- Network/transport errors throw `HttpException`.
- Validation issues throw `ValidationException`.
- All exceptions extend `SdkException` for convenient catching.

```php
use WebniamPayment\Exceptions\SdkException;

try {
    $client->createUser([
        'email' => 'customer@example.com',
        'name' => 'Customer Name',
    ]);
} catch (SdkException $e) {
    // inspect $e->getMessage()
}
```

## Testing Tips

Use the sandbox/test environment (`env => 'TEST'`) whenever possible. The SDK automatically uppercases `env` values and lowercases currencies/sources to match the backend expectations.
