<?php

namespace WebniamPayment;

use WebniamPayment\Exceptions\ValidationException;
use WebniamPayment\Http\CurlHttpClient;
use WebniamPayment\Http\HttpClientInterface;
use WebniamPayment\Response\ApiResponse;
use WebniamPayment\Validation\Validator;

class Client
{
    private string $baseUrl;
    private string $businessId;
    private string $apiKey;
    private HttpClientInterface $httpClient;

    public function __construct(string $baseUrl, string $businessId, string $apiKey, ?HttpClientInterface $httpClient = null)
    {
        $baseUrl = rtrim($baseUrl, " \t\n\r\0\x0B");
        if ($baseUrl === '') {
            throw new ValidationException('Base URL must be a non-empty string');
        }

        $this->baseUrl = $baseUrl;
        $this->businessId = $businessId;
        $this->apiKey = $apiKey;
        $this->httpClient = $httpClient ?? new CurlHttpClient();
    }

    public function getHttpClient(): HttpClientInterface
    {
        return $this->httpClient;
    }

    public function setHttpClient(HttpClientInterface $httpClient): void
    {
        $this->httpClient = $httpClient;
    }

    public function validateBusiness(?string $environment = null): ApiResponse
    {
        return $this->dispatch('validate_business', $this->buildEnvPayload($environment));
    }

    public function getBusinessInfo(?string $environment = null): ApiResponse
    {
        return $this->dispatch('get_business_info', $this->buildEnvPayload($environment));
    }

    public function getAllUsers(): ApiResponse
    {
        return $this->dispatch('get_all_users');
    }

    public function getStripePublicKey(string $environment = 'TEST'): ApiResponse
    {
        return $this->dispatch('get_stripe_public_key', ['env' => strtoupper($environment)]);
    }

    public function validateUser(?string $userId = null, ?string $email = null): ApiResponse
    {
        return $this->dispatch('validate_user', $this->withUser([], $userId, $email));
    }

    public function getUserInfo(?string $userId = null, ?string $email = null): ApiResponse
    {
        return $this->dispatch('get_user_info', $this->withUser([], $userId, $email));
    }

    public function getUserCards(?string $userId = null, ?string $email = null, string $source = 'both'): ApiResponse
    {
        $payload = $this->withUser(['source' => strtolower($source)], $userId, $email);
        return $this->dispatch('get_user_cards', $payload);
    }

    public function getUserPrimaryCard(?string $userId = null, ?string $email = null, string $source = 'both'): ApiResponse
    {
        $payload = $this->withUser(['source' => strtolower($source)], $userId, $email);
        return $this->dispatch('get_user_primary_card', $payload);
    }

    public function getUserWallet(?string $userId = null, ?string $email = null): ApiResponse
    {
        return $this->dispatch('get_user_wallet', $this->withUser([], $userId, $email));
    }

    /**
     * @param array{name?: string, email?: string} $updates
     */
    public function updateUser(string $userId, array $updates): ApiResponse
    {
        $payload = ['user_id' => $userId, 'updates' => $updates];
        return $this->dispatch('update_user', $payload);
    }

    /**
     * @param array{name: string, email: string, phone?: string} $newUser
     */
    public function createUser(array $newUser): ApiResponse
    {
        return $this->dispatch('create_user', ['new_user' => $newUser]);
    }

    /**
     * @param array{
     *     user_id?: string,
     *     email?: string,
     *     amount: float|int|string,
     *     currency?: string,
     *     source?: string,
     *     description?: string|null,
     *     env?: string
     * } $options
     */
    public function chargeCustomer(array $options): ApiResponse
    {
        $payload = $this->prepareChargePayload($options);
        return $this->dispatch('charge_customer', $payload);
    }

    /**
     * @param array{
     *     user_id?: string,
     *     email?: string,
     *     amount: float|int|string,
     *     currency?: string,
     *     source?: string,
     *     description?: string|null,
     *     env?: string
     * } $options
     */
    public function fundCustomer(array $options): ApiResponse
    {
        $payload = $this->prepareChargePayload($options);
        return $this->dispatch('fund_customer', $payload);
    }

    /**
     * @param array{user_id: string, token: string, env: string, email?: string, name?: string} $customerData
     */
    public function createStripeCustomer(array $customerData): ApiResponse
    {
        return $this->dispatch('create_stripe_customer', ['data' => $customerData]);
    }

    /**
     * @param array{
     *     user_id: string,
     *     email: string,
     *     card_number: string,
     *     exp_month: int|string,
     *     exp_year: int|string,
     *     card_code: string,
     *     env?: string
     * } $customerData
     */
    public function createAuthorizeCustomer(array $customerData): ApiResponse
    {
        return $this->dispatch('create_authorize_customer', ['data' => $customerData]);
    }

    public function getWalletHistory(?string $userId = null, ?string $email = null, int $limit = 50): ApiResponse
    {
        $payload = $this->withUser(['limit' => $limit], $userId, $email);
        return $this->dispatch('get_wallet_history', $payload);
    }

    public function chargeWallet(?string $userId, ?string $email, $amount, ?string $reason = null, ?string $description = null): ApiResponse
    {
        $payload = $this->withUser([
            'amount' => $amount,
            'reason' => $reason,
            'description' => $description,
        ], $userId, $email);

        return $this->dispatch('charge_wallet', $payload);
    }

    public function addWalletFunds(?string $userId, ?string $email, $amount, ?string $reason = null, ?string $description = null): ApiResponse
    {
        $payload = $this->withUser([
            'amount' => $amount,
            'reason' => $reason,
            'description' => $description,
        ], $userId, $email);

        return $this->dispatch('add_wallet_funds', $payload);
    }

    /**
     * @param array{
     *     source?: string,
     *     page?: int,
     *     limit?: int,
     *     status?: string,
     *     from_date?: string,
     *     to_date?: string
     * } $filters
     */
    public function getPaymentRecords(?string $userId, ?string $email, array $filters = []): ApiResponse
    {
        $payload = $this->withUser($filters, $userId, $email);
        return $this->dispatch('get_payment_records', $payload);
    }

    /**
     * @param array{
     *     currency?: string,
     *     source?: string,
     *     interval_type?: string,
     *     interval_value?: int,
     *     description?: string|null,
     *     auto_increment?: float|int|string,
     *     auto_decrement?: float|int|string,
     *     max_cycles?: int,
     *     start_at?: string,
     *     end_at?: string|null,
     *     timezone?: string
     * } $options
     */
    public function createRecurring(?string $userId, ?string $email, $amount, array $options = []): ApiResponse
    {
        $payload = $this->withUser(array_merge($options, ['amount' => $amount]), $userId, $email);
        return $this->dispatch('create_recurring', $payload);
    }

    public function listRecurring(?string $userId, ?string $email, int $page = 1, int $limit = 25): ApiResponse
    {
        $payload = $this->withUser(['page' => $page, 'limit' => $limit], $userId, $email);
        return $this->dispatch('list_recurring', $payload);
    }

    /**
     * @param array<string, mixed> $updates
     */
    public function updateRecurring(string $recurringId, array $updates): ApiResponse
    {
        $payload = array_merge($updates, ['recurring_id' => $recurringId]);
        return $this->dispatch('update_recurring', $payload);
    }

    public function pauseRecurring(string $recurringId): ApiResponse
    {
        return $this->dispatch('pause_recurring', ['recurring_id' => $recurringId]);
    }

    public function resumeRecurring(string $recurringId): ApiResponse
    {
        return $this->dispatch('resume_recurring', ['recurring_id' => $recurringId]);
    }

    public function cancelRecurring(string $recurringId): ApiResponse
    {
        return $this->dispatch('cancel_recurring', ['recurring_id' => $recurringId]);
    }

    public function getRecurringLogs(string $recurringId, int $limit = 50): ApiResponse
    {
        return $this->dispatch('get_recurring_logs', ['recurring_id' => $recurringId, 'limit' => $limit]);
    }

    /**
     * @param array{name: string, email: string, domain: string} $newBusiness
     */
    public function registerBusiness(array $newBusiness): ApiResponse
    {
        return $this->dispatch('register_business', ['new_business' => $newBusiness]);
    }

    /**
     * @param array{name?: string, email?: string} $updates
     */
    public function modifyBusiness(array $updates): ApiResponse
    {
        return $this->dispatch('modify_business', ['updates' => $updates]);
    }

    public function changeApiKey(?string $targetBusinessId = null): ApiResponse
    {
        $payload = [];
        if ($targetBusinessId !== null) {
            $payload['target_business_id'] = $targetBusinessId;
        }
        return $this->dispatch('change_api_key', $payload);
    }

    /**
     * @param array{
     *     env: string,
     *     stripe_secret_key: string,
     *     stripe_public_key: string,
     *     stripe_webhook_secret?: string,
     *     account_id?: string
     * } $credentials
     */
    public function addStripeCredentials(array $credentials): ApiResponse
    {
        return $this->dispatch('add_stripe_credentials', ['credentials' => $credentials]);
    }

    /**
     * @param array{
     *     env: string,
     *     stripe_secret_key?: string,
     *     stripe_public_key?: string,
     *     stripe_webhook_secret?: string,
     *     account_id?: string
     * } $credentials
     */
    public function modifyStripeCredentials(array $credentials): ApiResponse
    {
        return $this->dispatch('modify_stripe_credentials', ['credentials' => $credentials]);
    }

    /**
     * @param array{
     *     env: string,
     *     api_login_id: string,
     *     transaction_key: string,
     *     signature_key?: string
     * } $credentials
     */
    public function addAuthorizeCredentials(array $credentials): ApiResponse
    {
        return $this->dispatch('add_authorize_credentials', ['credentials' => $credentials]);
    }

    /**
     * @param array{
     *     env: string,
     *     api_login_id?: string,
     *     transaction_key?: string,
     *     signature_key?: string
     * } $credentials
     */
    public function modifyAuthorizeCredentials(array $credentials): ApiResponse
    {
        return $this->dispatch('modify_authorize_credentials', ['credentials' => $credentials]);
    }

    /**
     * @param array<string, mixed> $payload
     */
    private function dispatch(string $action, array $payload = []): ApiResponse
    {
        $requestPayload = $this->preparePayload($action, $payload);
        Validator::validate($action, $requestPayload);

        return $this->httpClient->post($this->baseUrl, $requestPayload);
    }

    /**
     * @param array<string, mixed> $payload
     * @return array<string, mixed>
     */
    private function preparePayload(string $action, array $payload): array
    {
        $envelope = [
            'action' => $action,
            'business_id' => $this->businessId,
            'api_key' => $this->apiKey,
        ];

        $normalized = $this->filterNull($payload);

        if (isset($normalized['description']) && is_string($normalized['description'])) {
            $normalized['description'] = $this->normalizeDescription($normalized['description']);
        }

        return array_merge($envelope, $normalized);
    }

    /**
     * @param array<string, mixed> $options
     * @return array<string, mixed>
     */
    private function prepareChargePayload(array $options): array
    {
        $payload = $this->withUser($options, $options['user_id'] ?? null, $options['email'] ?? null);

        if (isset($payload['source'])) {
            $payload['source'] = strtolower((string) $payload['source']);
        }

        if (isset($payload['currency'])) {
            $payload['currency'] = strtolower((string) $payload['currency']);
        }

        if (isset($payload['env'])) {
            $payload['env'] = strtoupper((string) $payload['env']);
        }

        if (isset($payload['description']) && is_string($payload['description'])) {
            $payload['description'] = $this->normalizeDescription($payload['description']);
        }

        return $payload;
    }

    /**
     * @param array<string, mixed> $payload
     * @return array<string, mixed>
     */
    private function withUser(array $payload, ?string $userId, ?string $email): array
    {
        if ($userId !== null) {
            $payload['user_id'] = $userId;
        }

        if ($email !== null) {
            $payload['email'] = $email;
        }

        return $this->filterNull($payload);
    }

    private function normalizeDescription(string $value): string
    {
        if ($this->isBase64($value)) {
            return $value;
        }

        return base64_encode($value);
    }

    private function isBase64(string $value): bool
    {
        if ($value === '') {
            return true;
        }

        $decoded = base64_decode($value, true);
        if ($decoded === false) {
            return false;
        }

        return base64_encode($decoded) === $value;
    }

    /**
     * @param array<string, mixed> $payload
     * @return array<string, mixed>
     */
    private function filterNull(array $payload): array
    {
        return array_filter(
            $payload,
            static fn ($value) => $value !== null
        );
    }

    /**
     * @return array<string, string>
     */
    private function buildEnvPayload(?string $environment): array
    {
        if ($environment === null) {
            return [];
        }

        return ['env' => strtoupper($environment)];
    }
}
