<?php

namespace WebniamPayment\Validation;

use WebniamPayment\Exceptions\ValidationException;

class Validator
{
    /**
     * @param array<mixed> $data
     */
    public static function validate(string $action, array $data): void
    {
        switch ($action) {
            case 'validate_business':
            case 'get_business_info':
            case 'get_all_users':
            case 'get_stripe_public_key':
                if (isset($data['env'])) {
                    self::ensureEnum($action, $data['env'], 'env', ['TEST', 'LIVE']);
                }
                return;

            case 'validate_user':
            case 'get_user_info':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                return;

            case 'get_user_cards':
            case 'get_user_primary_card':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                self::ensureEnum($action, $data['source'] ?? 'both', 'source', ['stripe', 'authorize', 'both']);
                return;

            case 'get_user_wallet':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                return;

            case 'update_user':
                self::ensureRequiredFields($action, $data, ['user_id', 'updates']);
                self::ensureArray($action, $data['updates'], 'updates');
                return;

            case 'create_user':
                self::ensureRequiredFields($action, $data, ['new_user']);
                self::ensureArray($action, $data['new_user'], 'new_user');
                $newUser = $data['new_user'];
                self::ensureRequiredFields($action, $newUser, ['email', 'name']);
                self::ensureEmail($action, $newUser['email'], 'new_user.email');
                return;

            case 'charge_customer':
            case 'fund_customer':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                self::ensureRequiredFields($action, $data, ['amount']);
                self::ensurePositiveNumber($action, $data['amount'], 'amount');
                self::ensureCurrencyCode($action, $data['currency'] ?? null, 'currency');
                if (isset($data['source'])) {
                    self::ensureEnum($action, $data['source'], 'source', ['stripe', 'authorize']);
                }
                if (isset($data['description'])) {
                    self::ensureBase64($action, $data['description'], 'description');
                }
                return;

            case 'create_stripe_customer':
                self::ensureRequiredFields($action, $data, ['data']);
                self::ensureArray($action, $data['data'], 'data');
                $payload = $data['data'];
                self::ensureRequiredFields($action, $payload, ['user_id', 'token', 'env']);
                self::ensureEnum($action, $payload['env'], 'env', ['TEST', 'LIVE']);
                if (!empty($payload['email'])) {
                    self::ensureEmail($action, $payload['email'], 'email');
                }
                return;

            case 'create_authorize_customer':
                self::ensureRequiredFields($action, $data, ['data']);
                self::ensureArray($action, $data['data'], 'data');
                $payload = $data['data'];
                self::ensureRequiredFields($action, $payload, ['user_id', 'email', 'card_number', 'exp_month', 'exp_year', 'card_code']);
                self::ensureEmail($action, $payload['email'], 'email');
                self::ensureIntegerRange($action, $payload['exp_month'], 'exp_month', 1, 12);
                self::ensureIntegerRange($action, $payload['exp_year'], 'exp_year', intval(date('Y')), intval(date('Y')) + 25);
                self::ensureEnum($action, $payload['env'] ?? 'TEST', 'env', ['TEST', 'LIVE']);
                return;

            case 'get_wallet_history':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                self::ensureIntegerRange($action, $data['limit'] ?? 50, 'limit', 1, 200);
                return;

            case 'charge_wallet':
            case 'add_wallet_funds':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                self::ensureRequiredFields($action, $data, ['amount']);
                self::ensurePositiveNumber($action, $data['amount'], 'amount');
                return;

            case 'get_payment_records':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                self::ensureEnum($action, $data['source'] ?? 'stripe', 'source', ['stripe', 'authorize']);
                self::ensureIntegerRange($action, $data['page'] ?? 1, 'page', 1);
                self::ensureIntegerRange($action, $data['limit'] ?? 25, 'limit', 1, 100);
                return;

            case 'create_recurring':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                self::ensureRequiredFields($action, $data, ['amount']);
                self::ensurePositiveNumber($action, $data['amount'], 'amount');
                self::ensureEnum($action, $data['source'] ?? 'stripe', 'source', ['wallet', 'stripe', 'authorize', 'both']);
                self::ensureEnum($action, $data['interval_type'] ?? 'month', 'interval_type', ['hour', '12h', 'day', 'week', 'month', '6month', 'year']);
                self::ensureIntegerRange($action, $data['interval_value'] ?? 1, 'interval_value', 1);
                self::ensureCurrencyCode($action, $data['currency'] ?? null, 'currency');
                return;

            case 'list_recurring':
                self::ensureOneOf($action, $data, ['user_id', 'email']);
                self::ensureIntegerRange($action, $data['page'] ?? 1, 'page', 1);
                self::ensureIntegerRange($action, $data['limit'] ?? 25, 'limit', 1, 100);
                return;

            case 'update_recurring':
                self::ensureRequiredFields($action, $data, ['recurring_id']);
                return;

            case 'pause_recurring':
            case 'resume_recurring':
            case 'cancel_recurring':
            case 'get_recurring_logs':
                self::ensureRequiredFields($action, $data, ['recurring_id']);
                if ($action === 'get_recurring_logs') {
                    self::ensureIntegerRange($action, $data['limit'] ?? 50, 'limit', 1, 200);
                }
                return;

            case 'register_business':
                self::ensureRequiredFields($action, $data, ['new_business']);
                self::ensureArray($action, $data['new_business'], 'new_business');
                $newBusiness = $data['new_business'];
                self::ensureRequiredFields($action, $newBusiness, ['name', 'email', 'domain']);
                self::ensureNonEmptyString($action, $newBusiness['name'], 'new_business.name');
                self::ensureEmail($action, $newBusiness['email'], 'new_business.email');
                self::ensureDomain($action, $newBusiness['domain'], 'new_business.domain');
                return;

            case 'modify_business':
                self::ensureRequiredFields($action, $data, ['updates']);
                self::ensureArray($action, $data['updates'], 'updates');
                self::ensureOneOf($action, $data['updates'], ['name', 'email']);
                if (isset($data['updates']['name'])) {
                    self::ensureNonEmptyString($action, $data['updates']['name'], 'updates.name');
                }
                if (isset($data['updates']['email'])) {
                    self::ensureEmail($action, $data['updates']['email'], 'updates.email');
                }
                return;

            case 'change_api_key':
                if (isset($data['target_business_id'])) {
                    self::ensureNonEmptyString($action, $data['target_business_id'], 'target_business_id');
                }
                return;

            case 'add_stripe_credentials':
                self::ensureRequiredFields($action, $data, ['credentials']);
                self::ensureArray($action, $data['credentials'], 'credentials');
                $creds = $data['credentials'];
                self::ensureRequiredFields($action, $creds, ['env', 'stripe_secret_key', 'stripe_public_key']);
                self::ensureEnum($action, $creds['env'], 'credentials.env', ['TEST', 'LIVE']);
                self::ensureNonEmptyString($action, $creds['stripe_secret_key'], 'credentials.stripe_secret_key');
                self::ensureNonEmptyString($action, $creds['stripe_public_key'], 'credentials.stripe_public_key');
                if (isset($creds['stripe_webhook_secret'])) {
                    self::ensureNonEmptyString($action, $creds['stripe_webhook_secret'], 'credentials.stripe_webhook_secret');
                }
                if (isset($creds['account_id'])) {
                    self::ensureNonEmptyString($action, $creds['account_id'], 'credentials.account_id');
                }
                return;

            case 'modify_stripe_credentials':
                self::ensureRequiredFields($action, $data, ['credentials']);
                self::ensureArray($action, $data['credentials'], 'credentials');
                $creds = $data['credentials'];
                self::ensureRequiredFields($action, $creds, ['env']);
                self::ensureEnum($action, $creds['env'], 'credentials.env', ['TEST', 'LIVE']);
                self::ensureOneOf($action, $creds, ['stripe_secret_key', 'stripe_public_key', 'stripe_webhook_secret', 'account_id']);
                if (isset($creds['stripe_secret_key'])) {
                    self::ensureNonEmptyString($action, $creds['stripe_secret_key'], 'credentials.stripe_secret_key');
                }
                if (isset($creds['stripe_public_key'])) {
                    self::ensureNonEmptyString($action, $creds['stripe_public_key'], 'credentials.stripe_public_key');
                }
                if (isset($creds['stripe_webhook_secret'])) {
                    self::ensureNonEmptyString($action, $creds['stripe_webhook_secret'], 'credentials.stripe_webhook_secret');
                }
                if (isset($creds['account_id'])) {
                    self::ensureNonEmptyString($action, $creds['account_id'], 'credentials.account_id');
                }
                return;

            case 'add_authorize_credentials':
                self::ensureRequiredFields($action, $data, ['credentials']);
                self::ensureArray($action, $data['credentials'], 'credentials');
                $creds = $data['credentials'];
                self::ensureRequiredFields($action, $creds, ['env', 'api_login_id', 'transaction_key']);
                self::ensureEnum($action, $creds['env'], 'credentials.env', ['TEST', 'LIVE']);
                self::ensureNonEmptyString($action, $creds['api_login_id'], 'credentials.api_login_id');
                self::ensureNonEmptyString($action, $creds['transaction_key'], 'credentials.transaction_key');
                if (isset($creds['signature_key'])) {
                    self::ensureNonEmptyString($action, $creds['signature_key'], 'credentials.signature_key');
                }
                return;

            case 'modify_authorize_credentials':
                self::ensureRequiredFields($action, $data, ['credentials']);
                self::ensureArray($action, $data['credentials'], 'credentials');
                $creds = $data['credentials'];
                self::ensureRequiredFields($action, $creds, ['env']);
                self::ensureEnum($action, $creds['env'], 'credentials.env', ['TEST', 'LIVE']);
                self::ensureOneOf($action, $creds, ['api_login_id', 'transaction_key', 'signature_key']);
                if (isset($creds['api_login_id'])) {
                    self::ensureNonEmptyString($action, $creds['api_login_id'], 'credentials.api_login_id');
                }
                if (isset($creds['transaction_key'])) {
                    self::ensureNonEmptyString($action, $creds['transaction_key'], 'credentials.transaction_key');
                }
                if (isset($creds['signature_key'])) {
                    self::ensureNonEmptyString($action, $creds['signature_key'], 'credentials.signature_key');
                }
                return;

            default:
                throw new ValidationException(sprintf('Unsupported action "%s"', $action));
        }
    }

    /**
     * @param array<mixed> $payload
     * @param array<int, string> $fields
     */
    private static function ensureRequiredFields(string $action, array $payload, array $fields): void
    {
        $missing = [];
        foreach ($fields as $field) {
            if (!array_key_exists($field, $payload) || $payload[$field] === null || $payload[$field] === '') {
                $missing[] = $field;
            }
        }

        if (!empty($missing)) {
            throw new ValidationException(sprintf('%s: missing required field(s): %s', $action, implode(', ', $missing)));
        }
    }

    /**
     * @param array<mixed> $payload
     * @param array<int, string> $fields
     */
    private static function ensureOneOf(string $action, array $payload, array $fields): void
    {
        foreach ($fields as $field) {
            if (array_key_exists($field, $payload) && $payload[$field] !== null && $payload[$field] !== '') {
                return;
            }
        }

        throw new ValidationException(sprintf('%s: at least one of the following fields must be provided: %s', $action, implode(', ', $fields)));
    }

    /**
     * @param array<int, string> $allowed
     */
    private static function ensureEnum(string $action, ?string $value, string $field, array $allowed, bool $caseInsensitive = true): void
    {
        if ($value === null) {
            return;
        }

        $check = $caseInsensitive ? strtolower($value) : $value;
        $valid = $caseInsensitive ? array_map('strtolower', $allowed) : $allowed;

        if (!in_array($check, $valid, true)) {
            throw new ValidationException(sprintf('%s: %s must be one of: %s', $action, $field, implode(', ', $allowed)));
        }
    }

    private static function ensurePositiveNumber(string $action, $value, string $field): void
    {
        if (!is_numeric($value)) {
            throw new ValidationException(sprintf('%s: %s must be a numeric value', $action, $field));
        }

        if (floatval($value) <= 0) {
            throw new ValidationException(sprintf('%s: %s must be greater than zero', $action, $field));
        }
    }

    private static function ensureIntegerRange(string $action, $value, string $field, int $min, ?int $max = null): void
    {
        if ($value === null) {
            return;
        }

        if (filter_var($value, FILTER_VALIDATE_INT) === false) {
            throw new ValidationException(sprintf('%s: %s must be an integer', $action, $field));
        }

        $int = intval($value);
        if ($int < $min || ($max !== null && $int > $max)) {
            $range = $max === null ? sprintf('%d+', $min) : sprintf('%d-%d', $min, $max);
            throw new ValidationException(sprintf('%s: %s must be within range %s', $action, $field, $range));
        }
    }

    private static function ensureEmail(string $action, ?string $email, string $field): void
    {
        if ($email !== null && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new ValidationException(sprintf('%s: invalid email format for %s', $action, $field));
        }
    }

    private static function ensureBase64(string $action, ?string $value, string $field): void
    {
        if ($value === null) {
            return;
        }

        if (base64_decode($value, true) === false) {
            throw new ValidationException(sprintf('%s: %s must be a base64 encoded string', $action, $field));
        }
    }

    private static function ensureArray(string $action, $value, string $field): void
    {
        if (!is_array($value)) {
            throw new ValidationException(sprintf('%s: %s must be an object', $action, $field));
        }
    }

    private static function ensureCurrencyCode(string $action, ?string $value, string $field): void
    {
        if ($value === null) {
            return;
        }

        if (!preg_match('/^[A-Za-z]{3}$/', $value)) {
            throw new ValidationException(sprintf('%s: %s must be a 3-letter currency code', $action, $field));
        }
    }

    private static function ensureNonEmptyString(string $action, $value, string $field): void
    {
        if (!is_string($value) || trim($value) === '') {
            throw new ValidationException(sprintf('%s: %s must be a non-empty string', $action, $field));
        }
    }

    private static function ensureDomain(string $action, ?string $value, string $field): void
    {
        if ($value === null) {
            return;
        }

        $value = strtolower(trim($value));
        if ($value === '') {
            throw new ValidationException(sprintf('%s: %s must be a non-empty domain', $action, $field));
        }

        if (!preg_match('/^(?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?)(?:\.(?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?))+$/i', $value)) {
            throw new ValidationException(sprintf('%s: invalid domain format for %s', $action, $field));
        }
    }
}
